-- Parameter für OCI Konfiguration
CREATE TABLE IF NOT EXISTS x_10_interfaces.oci_parameter (
    oci_id          SERIAL       PRIMARY KEY,
    oci_krz         varchar(21)  NOT NULL REFERENCES adk ON UPDATE CASCADE,  -- Lieferantenkürzel
    oci_name        varchar(50)  NOT NULL,                                   -- OCI Parametername
    oci_value       varchar(200) NOT NULL,                                   -- OCI Parameterwert
    oci_type        varchar(20)  NOT NULL,                                   -- OCI Parametertyp
    oci_description text,                                                    -- Beschreibung des Parameters
    oci_is_active   boolean      NOT NULL DEFAULT true,                      -- Aktiv/Inaktiv Flag
    CONSTRAINT oci_parameter_unique_krz_name UNIQUE (oci_krz, oci_name),
    CONSTRAINT oci_parameter_type_check CHECK (oci_type IN ('HOOK', 'LOGIN', 'FUNCTION', 'CONSTANT'))
);

-- Erstelle Index für häufige Abfragen
CREATE INDEX IF NOT EXISTS idx_oci_parameter_krz ON x_10_interfaces.oci_parameter(oci_krz);
CREATE INDEX IF NOT EXISTS idx_oci_parameter_active ON x_10_interfaces.oci_parameter(oci_is_active);

-- Dokumentation der Spalten
COMMENT ON TABLE x_10_interfaces.oci_parameter IS 'Speichert OCI-Konfigurationsparameter für Lieferanten-Kataloge';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_id IS 'Eindeutige ID des OCI-Parameters';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_krz IS 'Verweis auf das Lieferantenkürzel in der ADK-Tabelle';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_name IS 'Name des OCI-Parameters (z.B. HOOK_URL, USERNAME)';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_value IS 'Wert des OCI-Parameters';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_type IS 'Typ des Parameters: HOOK (Basis-URL), LOGIN (Anmeldedaten), CONSTANT (Konstante)';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_description IS 'Optionale Beschreibung des Parameters';
COMMENT ON COLUMN x_10_interfaces.oci_parameter.oci_is_active IS 'Flag ob der Parameter aktiv ist';


-- Validierung der OCI-URLs
CREATE OR REPLACE FUNCTION x_10_interfaces.oci__validate_url(
    IN _url varchar
  )
  RETURNS boolean AS $$
BEGIN
  RETURN _url ~ E'^https:\\/\\/[\\w\\-]+(\\.[\\w\\-]+)+([\\w\\-\\.,@?^=%&amp;:\\/~\\+#]*[\\w\\-\\@?^=%&amp;\\/~\\+#])?$';
END $$ LANGUAGE plpgsql;

-- Trigger für URL-Validierung und Verschlüsselung sensitiver Daten
CREATE OR REPLACE FUNCTION x_10_interfaces.oci_parameter_trigger_function()
  RETURNS trigger AS $$
BEGIN
  -- URL-Validierung für HOOK-Parameter
  IF NEW.oci_type = 'HOOK' AND NEW.oci_name = 'HOOK_URL' THEN
    IF NOT x_10_interfaces.oci__validate_url(NEW.oci_value) THEN
      RAISE EXCEPTION 'Ungültige URL: %. URLs müssen mit https:// beginnen.', NEW.oci_value;
    END IF;
  END IF;

  RETURN NEW;
END $$ LANGUAGE plpgsql;

-- Trigger für die oci_parameter Tabelle
DROP TRIGGER IF EXISTS oci_parameter_validation_trigger ON x_10_interfaces.oci_parameter;
CREATE TRIGGER oci_parameter_validation_trigger
  BEFORE INSERT OR UPDATE ON x_10_interfaces.oci_parameter
  FOR EACH ROW EXECUTE FUNCTION x_10_interfaces.oci_parameter_trigger_function();